#!/bin/bash
#
# Installs Docker Engine on Debian-based Linux distributions with colored output.
# Runs apt operations quietly, showing high-level status messages.
#
# Usage:
#   ./docker.sh
#   curl -fsSL https://cdn.albert.lol/docker.sh | bash
#
# Requires:
#   - Running on a Debian-based system (Debian, Ubuntu, etc.)
#   - User executing the script must have sudo privileges.
#   - Internet connection.
#   - A terminal that supports ANSI color codes.

# Strict mode
set -euo pipefail

# --- Configuration ---
readonly DOCKER_GPG_KEY_PATH="/etc/apt/keyrings/docker.asc"
readonly DOCKER_APT_SOURCE_PATH="/etc/apt/sources.list.d/docker.list"

# --- Colors ---
# Reference: https://misc.flogisoft.com/bash/tip_colors_and_formatting
readonly COLOR_RESET='\033[0m'
readonly COLOR_RED='\033[0;31m'
readonly COLOR_GREEN='\033[0;32m'
readonly COLOR_YELLOW='\033[0;33m'
readonly COLOR_BLUE='\033[0;34m'
readonly COLOR_BOLD='\033[1m'

# --- Helper Functions ---

# Logs an informational message (Blue).
# Arguments: $*: Message to log.
log_info() {
  echo -e "${COLOR_BLUE}[INFO]${COLOR_RESET} $*"
}

# Logs a success message (Green).
# Arguments: $*: Message to log.
log_success() {
  echo -e "${COLOR_GREEN}[SUCCESS]${COLOR_RESET} $*"
}

# Logs a warning message (Yellow).
# Arguments: $*: Message to log.
log_warning() {
 echo -e "${COLOR_YELLOW}[WARNING]${COLOR_RESET} $*"
}

# Logs an error message (Red) and exits.
# Arguments: $*: Message to log.
log_error() {
  # Ensure error message goes to stderr
  echo -e >&2 "${COLOR_RED}${COLOR_BOLD}[ERROR]${COLOR_RESET}${COLOR_RED} $*${COLOR_RESET}"
  exit 1
}

# Checks if a command exists.
# Arguments: $1: Command name.
command_exists() {
  command -v "$1" >/dev/null 2>&1
}

# --- Main Functions ---

# Step 1: Update package lists and install prerequisite packages.
install_prerequisites() {
  local pkgs=("ca-certificates" "curl" "gnupg")
  log_info "Updating package list (apt-get update)..."
  # Run update quietly (-qq)
  if ! sudo apt-get update -qq; then
      log_error "Failed to update package lists."
  fi

  log_info "Installing prerequisite packages: ${pkgs[*]}..."
  # Run install quietly (-qq) and assume yes (-y)
  if ! sudo apt-get install -y -qq "${pkgs[@]}"; then
      log_error "Failed to install prerequisite packages: ${pkgs[*]}."
  fi
  log_success "Prerequisites installed."
}

# Step 2: Set up Docker's official APT repository.
setup_apt_repository() {
  log_info "Setting up Docker's APT repository..."

  # Create the directory for the GPG key if it doesn't exist
  log_info "Ensuring keyring directory exists: /etc/apt/keyrings"
  sudo install -m 0755 -d "$(dirname "${DOCKER_GPG_KEY_PATH}")"

  # Download Docker's official GPG key
  log_info "Downloading Docker GPG key to ${DOCKER_GPG_KEY_PATH}"
  # Use curl with fail-fast, silent, show-error, location-follow flags
  # Capture curl stderr to check for errors even with -s
  local curl_stderr
  if ! curl_stderr=$(sudo curl -fsSL "https://download.docker.com/linux/debian/gpg" -o "${DOCKER_GPG_KEY_PATH}" 2>&1); then
      log_error "Failed to download Docker GPG key. Curl output: ${curl_stderr}"
  fi
  # Ensure the key is readable by apt
  sudo chmod a+r "${DOCKER_GPG_KEY_PATH}"
  log_success "Docker GPG key downloaded and permissions set."

  # Detect architecture and OS codename
  local arch
  arch="$(dpkg --print-architecture)"
  local codename
  # Source os-release safely
  if [[ -f /etc/os-release ]]; then
    # shellcheck source=/dev/null
    codename="$(. /etc/os-release && echo "$VERSION_CODENAME")"
  else
    log_error "Cannot determine OS codename. /etc/os-release not found."
  fi

  if [[ -z "$codename" ]]; then
      log_error "Could not determine VERSION_CODENAME from /etc/os-release."
  fi

  # Add the Docker repository to Apt sources
  log_info "Adding Docker repository source to ${DOCKER_APT_SOURCE_PATH}"
  echo \
    "deb [arch=${arch} signed-by=${DOCKER_GPG_KEY_PATH}] https://download.docker.com/linux/debian \
    ${codename} stable" | sudo tee "${DOCKER_APT_SOURCE_PATH}" > /dev/null
  log_success "Docker APT repository added."
}

# Step 3: Install Docker Engine packages.
install_docker() {
  local docker_pkgs=("docker-ce" "docker-ce-cli" "containerd.io" "docker-buildx-plugin" "docker-compose-plugin")
  log_info "Updating package list after adding Docker repository..."
   # Run update quietly (-qq)
  if ! sudo apt-get update -qq; then
      log_warning "Second 'apt-get update' failed. This might be okay if sources are correct, continuing installation..."
      # Don't exit here, as sometimes transient network issues cause this but install might still work if cache is okay.
  fi

  log_info "Installing Docker packages: ${docker_pkgs[*]}..."
   # Run install quietly (-qq) and assume yes (-y)
  if ! sudo apt-get install -y -qq "${docker_pkgs[@]}"; then
      log_error "Failed to install Docker packages: ${docker_pkgs[*]}."
  fi
  log_success "Docker packages installed."
}

# Step 4: Perform post-installation steps (add user to docker group).
post_install() {
  log_info "Adding current user (${USER}) to the 'docker' group..."
  if getent group docker > /dev/null; then
    # Use GID instead of name for group check in usermod, slightly more robust
    local docker_gid
    docker_gid=$(getent group docker | cut -d: -f3)
    if ! groups "$USER" | grep -qw "$docker_gid" && ! groups "$USER" | grep -qw "docker"; then
        if sudo usermod -aG docker "$USER"; then
            log_success "User '${USER}' added to the 'docker' group."
            log_warning "${COLOR_BOLD}IMPORTANT:${COLOR_RESET}${COLOR_YELLOW} You must log out and log back in for this change to take effect!${COLOR_RESET}"
            log_info "Alternatively, run 'newgrp docker' in your current shell (may require password)."
        else
            log_error "Failed to add user '${USER}' to the 'docker' group."
        fi
    else
        log_info "User '${USER}' is already in the 'docker' group."
    fi
  else
    log_warning "The 'docker' group does not seem to exist. Skipping adding user."
    log_warning "This might indicate an issue during Docker package installation."
 fi
}

# --- Main Execution ---

main() {
  log_info "${COLOR_BOLD}Starting Docker installation script...${COLOR_RESET}"

  # Basic checks
  if ! command_exists sudo; then
    log_error "'sudo' command not found. This script requires sudo privileges."
  fi
  # Allow running as root, but it's less common for interactive use.
  # if [[ $EUID -eq 0 ]]; then
  #    log_warning "Running as root. It's generally recommended to run as a user with sudo privileges."
  # fi
   if ! command_exists dpkg || ! command_exists apt-get; then
    log_error "'dpkg' or 'apt-get' not found. This script requires a Debian-based system."
   fi
   if ! [[ -f /etc/os-release ]]; then
     log_error "File /etc/os-release not found. Cannot determine OS version. Aborting."
   fi

  # Execute steps
  install_prerequisites
  setup_apt_repository
  install_docker
  post_install

  echo # Add a newline for spacing
  log_success "-----------------------------------------------------"
  log_success "${COLOR_BOLD}Docker installation completed successfully! 🎉${COLOR_RESET}"
  log_warning "${COLOR_BOLD}REMEMBER TO LOG OUT AND LOG BACK IN${COLOR_RESET}${COLOR_YELLOW} to use Docker without 'sudo'.${COLOR_RESET}"
  log_info "After logging back in, verify installation by running: ${COLOR_BOLD}docker run hello-world${COLOR_RESET}"
  log_success "-----------------------------------------------------"
}

# Run the main function if the script is executed directly
# This check prevents execution if the script is sourced, though unlikely for this use case.
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
  main
fi